/*
 * Copyright (C) 2011 AMIS research group, Faculty of Mathematics and Physics, Charles University in Prague, Czech Republic
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package cz.cuni.amis.pogamut.udk.experiments.impl;

import cz.cuni.amis.pogamut.base.utils.logging.LogCategory;
import cz.cuni.amis.pogamut.udk.communication.messages.gbcommands.DisconnectObserver;
import cz.cuni.amis.pogamut.udk.experiments.IExperiment;
import cz.cuni.amis.pogamut.udk.server.IUDKServer;
import cz.cuni.amis.pogamut.udk.utils.UCCWrapper;
import cz.cuni.amis.pogamut.udk.utils.UCCWrapper.UCCWrapperConf;
import cz.cuni.amis.utils.ExceptionToString;
import cz.cuni.amis.utils.flag.Flag;

/**
 * Helper class for implementing experiments
 * @author Martin Cerny
 */
public abstract class AbstractExperiment<RESULT, PARAMETERS> implements IExperiment<RESULT, PARAMETERS> {

    private Flag<Boolean> finished = new Flag<Boolean>(false);

    private RESULT result = null;
    
    private PARAMETERS parameters;

    private boolean succes = false;

    protected LogCategory log;

    protected UCCWrapper ucc;

    protected IUDKServer udkServer;
    
    protected String mapName;

    public AbstractExperiment(String mapName, PARAMETERS parameters) {
        this.mapName = mapName;
        this.parameters = parameters;
    }

    @Override
    public String getMapName() {
        return mapName;
    }

    
    

    @Override
    public void cleanup() {
        //to allow for GC
        log = null;
        ucc = null;
        udkServer = null;
    }

    @Override
    public String getDescription() {
        return toString();
    }

    @Override
    public Flag<Boolean> getFinished() {
        return finished;
    }

    @Override
    public final RESULT getResult() {
        return result;
    }

    @Override
    public PARAMETERS getParameters() {
        return parameters;
    }   
    

    /**
     * Sets the result and sets the finished flag to true
     * @throws NullPointerException if result is null.
     */
    protected void succeed(RESULT result) {
        if (result == null) {
            throw new NullPointerException("Result can't be set to null.");
        }
        this.result = result;
        this.succes = true;
        finished.setFlag(true);
    }


    @Override
    public void setServer(UCCWrapper ucc) {
        this.ucc = ucc;
        this.udkServer = ucc.getUTServer();
    }

    /**
     * Sets the UDK server - if {@link #setServer(cz.cuni.amis.pogamut.udk.utils.UCCWrapper) } is called,
     * this setting is overriden.
     * @param udkServer t
     */
    public void setUdkServer(IUDKServer udkServer) {
        this.udkServer = udkServer;
    }
    

    @Override
    public void setLog(LogCategory log) {
        this.log = log;
    }

    @Override
    public void startExperiment() {
    }

    @Override
    public boolean isSuccess() {
        return succes;
    }

    /**
     * Call to end the experiment with a failure
     * @param message might be null
     * @param e  exception taht caused the failure, might be nul
     */
    protected void fail(String message, Exception e) {
        String messageToLog;
        if (message != null) {
            messageToLog = "Experiment failed.";
        } else {
            messageToLog = "Experiment failed: " + message;
        }
        log.severe(messageToLog);
        if (e != null) {
            log.severe(ExceptionToString.process(e));            
        } 
        this.succes = false;
        finished.setFlag(true);

    }
    
    protected void fail(String message) {
        fail(message,null);
    }
    
    protected void fail(Exception ex){
        fail(null, ex);
    }

    protected void fail() {
        fail(null, null);
    }
}
