package cz.cuni.amis.pogamut.sposh.elements;

import java.awt.datatransfer.DataFlavor;
import java.util.List;

/**
 * Sense is basically a function in Python or Java. It is called and it returns some result.
 * This class is representation of sense, but it can also take an argument and compare it with predicate.
 *
 * During execution POSH engine calls the sense and compares return value
 * with argument by predicate. If no argument was passed, use true, if no predicate
 * was passed, use ==.
 *
 * @author Honza
 */
public class Sense extends PoshDummyElement implements Comparable<Sense> {

    /**
     * Various predicates used in <tt>Sense</tt>
     * @author Honza
     */
    public enum Predicate {

        EQUAL(new String[]{"==", "="}),
        NOT_EQUAL("!="),
        LOWER("<"),
        GREATER(">"),
        LOWER_OR_EQUAL("<="),
        GREATER_OR_EQUAL(">="),
        DEFAULT("==");				// default is "=="
        private String[] stringForm = null;

        private Predicate(String form) {
            stringForm = new String[]{form};
        }

        private Predicate(String[] form) {
            stringForm = form;
        }

        /**
         * Get Predicate enum from passed string
         * @param str passed predicate in string form
         * @return Predicate
         */
        public static Predicate getPredicate(String str) {
            if (str == null)
                return Predicate.DEFAULT;

            str = str.trim();

            for (Predicate p : Predicate.values()) {
                for (String form : p.stringForm) {
                    if (form.equals(str)) {
                        return p;
                    }
                }
            }
            throw new IllegalArgumentException("String \"" + str + "\" is not a predicate.");
        }

        /**
         * Get integer id of predicate. Used because of stupid combo box property pattern in NB.
         * @return
         */
        public int getId() {
            for (int i = 0; i < Predicate.values().length; i++) {
                if (values()[i] == this) {
                    return i;
                }
            }
            throw new RuntimeException("Predicate \"" + this.toString() + "\" wasn't found in list of predicates.");
        }

        @Override
        public String toString() {
            return stringForm[0];
        }
    }

    /**
     * This class specifies call to the sense. To do that, it requires
     * <ul>
     *  <li>name of sense</li>
     *  <li>list of parameters</li>
     * <ul>
     */
    public static class SenseCall {

        private final String name;
        private final CallParameters parameters;

        /**
         * This is a sensecall without parameters
         * @param name
         */
        protected SenseCall(String name) {
            this.name = name;
            this.parameters = new CallParameters();
        }
        protected SenseCall(String name, CallParameters parameters) {
            this.name = name;
            this.parameters = new CallParameters(parameters);
        }

        public String getName() {
            return name;
        }

        /**
         * @return unmodifiable list of parameters in correct order.
         */
        public CallParameters getParameters() {
            return parameters;
        }

        @Override
        public String toString() {
            return name;
        }
    }

    /**
     * Name of primitive that will be called along with list of parameters
     */
    private SenseCall senseCall;
    /**
     * When evaluating result of sense, should I compare value returned by the primitive
     * or should I just evaluate primtive and return?
     */
    private boolean compare = true;
    /**
     * Comparator used for comparison of value returned by primitive and operand
     */
    private Predicate _predicate = Predicate.DEFAULT;
    /**
     * Value can be number, string or nil
     * This can be null.
     */
    private Object operand = true;

    /**
     * Create sense that will evaluate true based on result of called primitive.
     * The primitive has no parameters.
     * @param senseName name of primitive
     */
    public Sense(String senseName) {
        this(new SenseCall(senseName));
    }

    /**
     * Create sense that will evaluate based on result of called primitive
     * some parameters
     * @param senseCall details of call primitive
     */
    public Sense(Sense.SenseCall senseCall) {
        this.senseCall = senseCall;
        this.compare = false;
    }

    /**
     * Create a Sense that takes a result of function and compares it with argument.
     * @param senseName name of function.
     * @param value number, string or nil
     * @param predicate string representation of predicate, same as in enum <tt>Predicate</tt>,
     *                  if null, Predicate.DEFAULT will be used.
     */
    public Sense(String senseName, String value, String predicate) {
        this(
                new SenseCall(senseName),
                Result.parseValue(value),
                Predicate.getPredicate(predicate)
                );
    }

    Sense(String senseName, String value, Predicate predicate) {
        this(
                new SenseCall(senseName),
                Result.parseValue(value),
                predicate);
    }

    /**
     *
     * @param senseCall
     * @param operand operand that will be parsed and changed into proper object
     * @param predicate
     */
    public Sense(SenseCall senseCall, String operand, Predicate predicate) {
        this(senseCall, Result.parseValue(operand), predicate);
    }

    private Sense(SenseCall senseCall, Object operand, Predicate predicate) {
        assert predicate!= null;

        this.senseCall = senseCall;
        this.compare = true;
        this.operand = operand;
        this._predicate = predicate;
    }

    public static final String psSenseName = "paSenseName";
    public static final String psPredicateIndex = "paPredicate";
    public static final String psValue = "paValue";
    public static final String psType = "paType";
    /*
     */

    public String getSenseName() {
        return senseCall.getName();
    }

    /**
     * Get param
     */
    public SenseCall getSenseCall() {
        return senseCall;
    }
    
    /**
     * Set new sense name if new name matches <tt>IDENT_PATTERN</tt>, name will be trimmed.
     * @param newSenseName
     */
    public void setSenseName(String newSenseName) {
        newSenseName = newSenseName.trim();

        if (newSenseName.matches(IDENT_PATTERN)) {
            String oldSenseName = this.senseCall.getName();

            this.senseCall = new SenseCall(newSenseName, senseCall.getParameters());
            this.firePropertyChange(Sense.psSenseName, oldSenseName, newSenseName);
        }
    }

    /**
     * Used in Node.Properties
     * XXX: Do more properly with custom editor
     * @return
     */
    public String getValueString() {
        return operand == null ? "null" : operand.toString();
    }

    /**
     * Set value of argument.
     * Used in Node.Properties
     * XXX: Do more properly with custom editor
     */
    public void setValueString(String newValueString) {
        compare = true;
        operand = Result.parseValue(newValueString);
        firePropertyChange(Sense.psValue, null, operand == null ? "null" : operand.toString());
    }

    /**
     * Get operand that should be used for evaluating this sense.
     * @see
     * @return
     */
    public Object getOperand() {
        return operand;
    }

    /**
     * Set value of argument (as class, e.g. Boolean, String, Integer...).
     *
     * @param newValue number, string or nil
     */
    public void setOperand(Object newValue) {
        firePropertyChange(Sense.psValue, operand, operand = newValue);
    }

    public Integer getPredicateIndex() {
        return this._predicate.getId();
    }

    public Predicate getPredicate() {
        return this._predicate;
    }

    /**
     * Set the predicate in the sense. If set to DEFAULT predicate, it won't be shown.
     * @param newPredicate non null
     */
    public void setPredicate(Predicate newPredicate) {
        compare = true;
        this._predicate = newPredicate;
        this.firePropertyChange(Sense.psPredicateIndex, null, newPredicate.getId());
    }

    /**
     * Set predicate, use index to array Predicate.values()[index]
     * @param newPredicateIndex index to new predicate
     */
    public void setPredicateIndex(Integer newPredicateIndex) {
        if (newPredicateIndex != null) {
            this._predicate = Predicate.values()[newPredicateIndex];
            this.firePropertyChange(Sense.psPredicateIndex, null, newPredicateIndex);
        } else {
            this._predicate = Predicate.DEFAULT;
            this.firePropertyChange(Sense.psPredicateIndex, null, Predicate.DEFAULT.getId());
        }
    }

    /**
     * Take other sense and change all attributes in the sense to be same
     * as in other sense.
     * @param other
     */
    public void changeTo(Sense other) {
        this.compare = other.compare;
        this.setSenseName(other.getSenseName());
        this.setOperand(other.getOperand());
        this.setPredicateIndex(other.getPredicateIndex());
    }

    /**
     * Return text representation in posh friendly manner.
     * Examples:
     *  (senseName)
     *  (senseName 2 !=)
     *  (senseName "Ahoj")
     * @return
     */
    @Override
    public String toString() {
        String res = "(" + senseCall;

        if (compare) {
            res += " " + operand;
            if (_predicate != Predicate.DEFAULT) {
                res += " " + _predicate;
            }
        }

        return res + ")";
    }

    @Override
    public String getDisplayName() {
        return getRepresentation();
    }

    @Override
    public List<PoshElement> getChildDataNodes() {
        return emptyChildrenList;
    }

    @Override
    public boolean moveChild(PoshElement child, int relativePosition) {
        return false;
    }
    public static final DataFlavor dataFlavor = new DataFlavor(Sense.class, "sense");

    @Override
    public DataFlavor getDataFlavor() {
        return dataFlavor;
    }

    @Override
    public void addChildDataNode(PoshElement newChild) {
        throw new RuntimeException("Class " + newChild.getClass().getSimpleName() + " not accepted.");
    }

    @Override
    public void neutralizeChild(PoshElement childNode) {
        throw new RuntimeException("Sense doesn't have children.");
    }

    /**
     * Get human readable representation of this sense.
     * @return
     */
    public String getRepresentation() {
        StringBuilder representation = new StringBuilder(getSenseName());

        if (compare) {
            representation.append(_predicate);
            representation.append(operand);
        }
        return representation.toString();
    }

    @Override
    public int compareTo(Sense o) {
        return this.getRepresentation().compareTo(o.getRepresentation());
    }
}