package cz.cuni.amis.pogamut.sposh.elements;

import java.util.AbstractList;
import java.util.ArrayList;
import java.util.List;

/**
 * This class is used as holder of all formal parameters for C or AP.
 * <p>
 * Specialized class is used instead of some Collection, because I want to
 * keep order in which parameters were added, but also throw error on duplicates.
 * 
 * @author Honza
 */
final public class FormalParameters extends AbstractList<FormalParameters.Parameter> {

    /**
     * Copy constructor.
     * @param parameters
     */
    FormalParameters(FormalParameters parameters) {
        this.parameters = new ArrayList<Parameter>();
        for (FormalParameters.Parameter param : parameters) {
            this.add(param);
        }
    }

    /**
     * This class represents one formal parameter of C or AP.
     * Senses and actions don't have formal parameters per se, because they are not
     * declared in the plan, but even they can be passed parameters.
     * <p>
     * Formal parameters are typeless variables that always have default value that
     * is used when caller "forgets" to specify the parameter.
     * @author Honza
     */
    final public static class Parameter {

        final private String name;
        final private Object defaultValue;

        /**
         * Create new formal parameter for C or AP.
         * @param name name of parameter, always start with "$"
         * @param defaultValue value that is used when no value is passed for this parameter
         */
        protected Parameter(String name, Object defaultValue) {
            this.name = name;
            this.defaultValue = defaultValue;
        }

        /**
         * @return the name of parameter
         */
        public String getName() {
            return name;
        }

        /**
         * @return the defaultValue of parameter
         */
        public Object getDefaultValue() {
            return defaultValue;
        }

        @Override
        public String toString() {
            return getName() + "=" + getDefaultValue();
        }
    }
    private List<Parameter> parameters;

    protected FormalParameters() {
        parameters = new ArrayList<Parameter>();
    }

    @Override
    public Parameter get(int index) {
        return parameters.get(index);
    }

    @Override
    public int size() {
        return parameters.size();
    }

    @Override
    public Parameter set(int index, Parameter element) {
        for (Parameter param : parameters) {
            if (param.getName().equals(element.getName()) && !param.equals(element)) {
                throw new IllegalArgumentException("Parameter with name \"" + element.getName() + "\" has already been specified in formal parameters.");
            }
        }
        return parameters.set(index, element);
    }

    @Override
    public void add(int index, Parameter element) {
        //Check that parameter with same name hasn't already been inserted
        for (Parameter param : parameters) {
            if (param.getName().equals(element.getName())) {
                throw new IllegalArgumentException("Parameter with name \"" + element.getName() + "\" has already been specified in formal parameters.");
            }
        }
        parameters.add(index, element);
    }

    @Override
    public Parameter remove(int index) {
        return parameters.remove(index);
    }

    /**
     * Does this list of formal parameters contains variable with name
     * variableName?
     * @param variableName variableName we are checking.
     * @return true if it contais variable with same name.
     */
    public boolean containsVariable(String variableName) {
        for (FormalParameters.Parameter param : parameters) {
            if (param.getName().equals(variableName)) {
                return true;
            }
        }
        return false;
    }


    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder("");

        for (int i=0; i<parameters.size();++i) {
            String parameterName = parameters.get(i).toString();
            if (i==0) {
                sb.append(parameterName);
            } else {
                sb.append(", " + parameterName);
            }
        }
        return sb.toString();
    }
}
