package cz.cuni.amis.pogamut.sposh.elements;

import java.lang.reflect.Array;
import java.util.regex.Pattern;

/**
 * Class representing value in posh plan.
 * @author Honza
 */
public class Result {

    /**
     * Is the value false?
     * null is false
     * Boolean false is false
     * Number 0 is false
     * Arrays of size 0 is false,
     * otherwise value is true
     * 
     * @see http://docs.python.org/library/stdtypes.html#truth-value-testing
     * @return true if value is null, value convertable to number and 0
     */
    public static boolean isFalse(Object value) {
        if (value == null) {
            return true;
        }
        if (isBoolean(value)) {
            return !getBoolean(value);
        }
        if (isNumber(value)) {
            return getNumber(value).doubleValue() == 0;
        }
        if (value.getClass().isArray()) {
            return Array.getLength(value) == 0;
        }
        return false;
    }

    /**
     * Is value true? Basically negate {@link Result#isFalse(java.lang.Object) }.
     * @param value value for which we want the information.
     */
    public static boolean isTrue(Object value) {
        return !isFalse(value);
    }

    public static boolean isNumber(Object value) {
        return value instanceof Number;
    }

    public static Number getNumber(Object value) {
        assert isNumber(value);
        return (Number) value;
    }

    /**
     * Is value instance of {@link Boolean} class?
     * @param value value to be tested
     * @return true if it is instance of {@link Boolean}
     */
    public static boolean isBoolean(Object value) {
        return value instanceof Boolean;
    }

    /**
     * Get boolean value from {@link Boolean} class.
     * @param value (value must be instance of {@link Boolean) class.
     * @return value of boolean.
     */
    public static boolean getBoolean(Object value) {
        assert isBoolean(value);
        return ((Boolean) value).booleanValue();
    }

    /**
     * Parse string from posh plan and convert it to object.
     * Rules of parsing:
     * "nil" - null
     * "true"/"false" - boolean
     * int number - integer
     * double number - double
     * otherwise string
     *
     * @param valueString string that will be parsed to object NOT NULL.
     * @return created object
     */
    public static Object parseValue(String valueString) {
        assert valueString != null;

        if ("nil".equalsIgnoreCase(valueString)) {
            return null;
        }

        if (Pattern.compile("([Tt]rue)|([Ff]alse)").matcher(valueString).matches()) {
            return Boolean.parseBoolean(valueString);
        }

        try {
            return Integer.parseInt(valueString);
        } catch (NumberFormatException ex) {
        }

        try {
            return Double.parseDouble(valueString);
        } catch (NumberFormatException ex) {
        }

        return valueString;
    }

    /**
     * Get numerical representation for object derived from class {@link Number}
     * and from {@link Boolean} (true = 1, false = 0)
     * @param value
     * @return numerical value of value.
     */
    public static double getNumerical(Object value) {
        if (isBoolean(value))
            return getBoolean(value) ? 1 : 0;

        return getNumber(value).doubleValue();
    }

    /**
     * Is value numerical(either number or boolean)?
     * @param value value to check if it is numerical
     */
    public static boolean isNumerical(Object value) {
        return isNumber(value) || isBoolean(value);
    }

    /**
     * Are two operands equal? Comparison is based on rules outlines in
     * http://docs.python.org/reference/expressions.html#notin
     * @param operand1
     * @param operand2
     * @return
     */
    public static boolean equal(Object operand1, Object operand2) {
        // I can compare them numericly
        if (isNumerical(operand1) && isNumerical(operand2)) {
            double op1 = getNumerical(operand1);
            double op2 = getNumerical(operand2);
            return op1 == op2;
        }
        // otherwise use standard equals
        return operand1 == null ? operand2 == null : operand1.equals(operand2);
    }

    /**
     * What is the comparison of operand1 and operand2?
     * @param operand1
     * @param operand2
     * @return a negative integer, zero, or a positive integer as this object is less than, equal to, or greater than the specified object. 
     */
    public static int compare(Object operand1, Object operand2) {
        // I can compare them numericly
        if (isNumerical(operand1) && isNumerical(operand2)) {
            double op1 = getNumerical(operand1);
            double op2 = getNumerical(operand2);
            int sig = (int)Math.signum(op1 - op2);
            return sig;
        }
        // otherwise use comparable
        if (operand1 == null) {
            if (operand2 == null) {
                return 0;
            }
            // two objects are not equal, but HOW? we have null and something.
            throw new IllegalArgumentException("I can't compare " + operand1 + " with " + operand2);
        }

        Comparable op1 = (Comparable) operand1;
        Comparable op2 = (Comparable) operand2;

        return op1.compareTo(op2);
    }
}
