package cz.cuni.amis.pogamut.sposh.engine;

import cz.cuni.amis.pogamut.sposh.elements.Result;
import cz.cuni.amis.pogamut.sposh.elements.Sense;
import cz.cuni.amis.pogamut.sposh.executor.IWorkExecutor;
import java.util.logging.Logger;

/**
 * This is used to execute single primitive, like in DE or C.
 * @author Honza
 */
class PrimitiveExecutor extends AbstractExecutor implements ElementExecutor {
    final private Sense.SenseCall actionCall;
    final private FireResult.Type successResult;
    final private FireResult.Type failResult;

    /**
     * Create a primitive executor that will call a specified action
     * within context of parameters. Always return SURFACE, whether the primitive fails or not.
     *
     * @param actionCall actionCall to primitive
     * @param ctx Context of this primitive. Shoudl differ from parent, but isn't necessary
     * @param log logger to record actions of this executor, can be null
     */
    PrimitiveExecutor(Sense.SenseCall actionCall, VariableContext ctx, Logger log) {
        this(actionCall, FireResult.Type.SURFACE, FireResult.Type.SURFACE, ctx, log);
    }

    /**
     * Create a primitive executor that will call a specified action
     * within context of parameters
     * 
     * @param actionCall actionCall to primitive
     * @param successResult what to return
     * @param failResult
     * @param ctx Context of this primitive. Shoudl differ from parent, but isn't necessary
     * @param log logger to record actions of this executor, can be null
     */
    PrimitiveExecutor(
            Sense.SenseCall actionCall,
            FireResult.Type successResult,
            FireResult.Type failResult,
            VariableContext ctx,
            Logger log) {
        super(ctx, log);
        this.actionCall = actionCall;
        this.successResult = successResult;
        this.failResult = failResult;
    }


    /**
     * Fire the action and return FireResult(false), so don't continue
     * the execution.
     * @param workExecuter
     * @return failResult if result of primitive is empty or false, true otherwise
     */
    @Override
    public FireResult fire(IWorkExecutor workExecuter) {
        Object result = workExecuter.executePrimitive(actionCall.getName(), ctx);
        if (Result.isFalse(result)) {
            return new FireResult(failResult);
        }
        return new FireResult(successResult);
    }

    @Override
    public String toString() {
        return this.getClass().getSimpleName() + "[" + actionCall.getName() + "]";
    }
}
