package cz.cuni.amis.pogamut.sposh.elements;

import cz.cuni.amis.pogamut.sposh.engine.VariableContext;
import java.util.AbstractList;
import java.util.ArrayList;
import java.util.List;

/**
 * This class is used to store parameters that were declared in plan during
 * call of C/AP/Sense/Action
 * @author Honza
 */
public class CallParameters extends AbstractList<CallParameters.Parameter> {

    public static abstract class Parameter {
        final private String parameterName;

        /**
         * Create paramter with name.
         * @param parameterName
         */
        protected Parameter(String parameterName) {
            this.parameterName = parameterName;
        }

        /**
         * Retrieve value of parameter.
         * @return
         */
        public abstract Object getValue();

        /**
         * Is value of this parameter retrieved from variable or value?
         * @return null if this is a value, string with name of parameter variable else
         */
        public abstract String getParameterVariable();

        public String getParameterName() {
            return parameterName;
        }
    }

    /**
     * This parameter is dependant on value of some kind of variable.
     */
    protected static class VariableParemeter extends CallParameters.Parameter {
        protected final String variableName;

        protected VariableParemeter(int index, String variableName) {
            super(Integer.toString(index));
            this.variableName = variableName;
        }

        /**
         * New call parameter.
         * @param parameterName name of parameter of called primitive
         * @param variableName variable where is stored the value passed to primitive
         */
        protected VariableParemeter(String parameterName, String variableName) {
            super(parameterName);
            this.variableName = variableName;
        }

        @Override
        public String getValue() {
            throw new UnsupportedOperationException("Not yet implemented");
        }

        @Override
        public String getParameterVariable() {
            return variableName;
        }
    }

    /**
     * This parameter is static value, neverchanging.
     */
    final protected static class ValueParameter extends CallParameters.Parameter {
        private final Object value;

        /**
         * Create a value parameter that represents fixed value.
         * <p>
         * Sequence number is used as name of this parameter. Since normal variables
         * starts with $, it won't mix.
         * 
         * @param sequenceNumber number of this parameter in sequence of all parameters. Starting from 0.
         * @param value value of parameter.
         */
        protected ValueParameter(int sequenceNumber, Object value) {
            super(Integer.toString(sequenceNumber));
            this.value = value;
        }

        protected ValueParameter(String parameterName, Object value) {
            super(parameterName);
            this.value = value;
        }

        @Override
        public Object getValue() {
            return value;
        }

        @Override
        public String getParameterVariable() {
            return null;
        }
    }

    
    
    private List<CallParameters.Parameter> parameters = new ArrayList<Parameter>();

    /**
     * Create a new list of call parameters.
     * <p>
     * Every added variable parameter has to be checked against list of formal
     * parameters.
     */
    protected CallParameters() {
    }

    /**
     * Copy constructor. Beware, this doesn't copy reference to the formal parameters.
     * @param parameters orginal
     */
    protected CallParameters(CallParameters parameters) {
        for (int i=0; i<parameters.size();++i) {
            this.parameters.add(parameters.get(i));
        }
    }


    @Override
    public synchronized CallParameters.Parameter get(int index) {
        return parameters.get(index);
    }

    @Override
    public synchronized int size() {
        return parameters.size();
    }

    public synchronized boolean addFormal(CallParameters.Parameter element, FormalParameters formalParams) {
        // check if variable is defined in the context
        String parameterVariable = element.getParameterVariable();
        if (parameterVariable != null) {
            if (!formalParams.containsVariable(parameterVariable)) {
                throw new IllegalArgumentException("Variable \"" + parameterVariable + "\" is not defined in formal parameters (" + formalParams.toString() + ").");
            }
        }

        // next check that named parameter isn't duplicated
        for (int i=0; i<parameters.size();++i) {
            String parameterName = parameters.get(i).getParameterName();
            // check if there isn't already variable name with same name
            if (parameterName != null && parameterName.equals(element.getParameterName())) {
                throw new IllegalArgumentException("Named parameter \"" + element.getParameterName() + "\" has already been defined.");
            }
        }
        return parameters.add(element);
    }

}
