package cz.cuni.amis.pogamut.sposh.elements;

import cz.cuni.amis.pogamut.sposh.elements.SolTime.TimeUnits;
import java.util.ArrayList;
import java.util.List;
import java.awt.datatransfer.DataFlavor;
import java.util.Collections;
import javax.swing.JOptionPane;

/**
 * AP is named list of actions, nothing more. It can be used instead as shortcut.
 * <pre>(AP avoid (minutes 10) (stop-bot rotate then-walk)) </pre>
 
 * Action patterns: or simple sequences.  These are a basic kind of plan aggregate
 * which turn out to be useful in quite a lot of situations, despite their lack of
 * flexibility.  They reduce the combinatorial complexity of the agent when a full
 * competence is not really necessary.
 * 
 * @author Honza
 */
public class ActionPattern extends PoshDummyElement implements Comparable<ActionPattern> {

	String _name;
	SolTime _timeout;
	ElementList<TriggeredAction> _actions;
	String _comment;
	public static final String apName = "apName";
	public static final String apTimeoutAmmount = "apTimeoutAmmount";
	public static final String apTimeoutUnits = "apTimeoutUnits";
	public static final String apComment = "apComment";
	
	public ActionPattern(String name) {
		this(name, new SolTime(), new ElementList<TriggeredAction>(new TriggeredAction[]{new TriggeredAction("do_nothing")}), "");
	}

	/**
	 * Create new AP
	 *
	 * @param name Name of AP
	 * @param solTime can be null
	 * @param ap List of actions
	 * @param comment
	 */
	ActionPattern(String name, SolTime solTime, ElementList<TriggeredAction> ap, String comment) {
		this._name = name;
		this._timeout = solTime == null ? new SolTime() : solTime;
		
		this._actions = ap;
		this._comment = comment;

		for (TriggeredAction action : ap) {
                    action.setParent(this);
                }
	}

	/**
	 * Add new TriggeredAction as child of this AP, take care of adding widget and so on.
	 * @param action
	 */
	public boolean addTriggeredAction(TriggeredAction action) {
		action.setParent(this);
		this._actions.add(action);

		PoshPlan root = getRootNode();
		if (root != null && root.isCycled()) {
			this._actions.remove(action);

			JOptionPane.showMessageDialog(
                                null,
                                "This action is causing cycle, please use another name.",
                                "Cycle detected",
                                JOptionPane.OK_OPTION);

			return false;
		} else {
			emitChildNode(action);
			return true;
		}
	}
	
	@Override
	public String toString() {
		String ret = "\t(AP " + _name;
		
		if (_timeout != null) {
			ret += " " + _timeout.toString();
		}
		ret += " " + _actions.toString();
		
		if (_comment != null && _comment.length() > 0) {
			ret += " \"" + _comment + '"';
		}
		
		return ret + ")\n";
	}

	@Override
	public List<PoshElement> getChildDataNodes() {
		return new ArrayList<PoshElement>(_actions);
	}

	/**
	 * Set comment of AP.
	 * @param comment text without " in it.
	 */
	public void setNodeComment(String comment) {
		if (comment != null && comment.contains("\"")) {
			return;
		}
                String oldComment = _comment;
		this._comment = comment;
		firePropertyChange(apComment, oldComment, comment);
	}

	public String getNodeComment() {
		return this._comment;
	}
	
	public Double getNodeTimeoutAmmount() {
		return this._timeout.getCount();
	}

	/**
	 * Set how many timeout units should AP wait until it gives up.
	 * @param a
	 */
	public void setNodeTimeoutAmmount(Double a) {
		if (a != null) {
			this._timeout.setCount(a);
			firePropertyChange(apTimeoutAmmount, null, a);
		}
	}

	public Integer getNodeTimeoutUnits() {
		return this._timeout.getUnits().getId();
	}

	/**
	 * Set type of units of timeout.
	 * @param newUnitsId <tt>getId()</tt> from enum <tt>TimeUnits</tt>.
	 */
	public void setNodeTimeoutUnits(Integer newUnitsId) {
		if (newUnitsId != null) {
			for (TimeUnits unit : TimeUnits.values()) {
				if (unit.getId() == newUnitsId) {
					_timeout.setUnits(unit);
					firePropertyChange(apTimeoutUnits, null, unit);
				}
			}
		}
	}

	/**
	 * Set name of AP. Check for cycles and make sure the name is unique
         * not same as some AP or Competence.
         *
	 * @param name string tham matches <tt>IDENT_PATTERN</tt>
	 */
	public void setNodeName(String name) throws ParseException {
		name = name.trim();

		if (name.matches(IDENT_PATTERN)) {
			if (!this.getNodeName().equals(name)) {
				if (getRootNode() != null && !getRootNode().isUniqueAPorComp(name)) {
					String msg  = "New name for action pattern '" + this._name + "'(" + name + ") is not unique for reaction plan.";
					JOptionPane.showMessageDialog(null, msg, "Duplicate name", JOptionPane.OK_OPTION);

					return;
				}
			}

			String oldName = this._name;
			this._name = name;

			if (getRootNode() != null && getRootNode().isCycled()) {
				this._name = oldName;

				String msg = "New name (" + name + ") for action pattern '" + this.getNodeName() + "' is causing cycle.";
        			JOptionPane.showMessageDialog(null, msg, "Cycle detected", JOptionPane.OK_OPTION);

				return;
			}

			firePropertyChange(apName, null, name);
		}
	}

	/**
	 * Get name of AP node.
	 * @return name of the node
	 */
	public String getNodeName() {
		return this._name;
	}

        @Override
	public String getDisplayName() {
		return this.getNodeName();
	}
	
	@Override
	public boolean moveChild(PoshElement child, int relativePosition) {
		return moveNodeInList(this._actions, child, relativePosition);
	}
	public static final DataFlavor dataFlavor = new DataFlavor(ActionPattern.class, "action-pattern-node");
	
	@Override
	public DataFlavor getDataFlavor() {
		return dataFlavor;
	}

	@Override
	public void addChildDataNode(PoshElement newChild) {
		if (newChild instanceof TriggeredAction) {
			this.addTriggeredAction((TriggeredAction) newChild);
		} else {
			throw new RuntimeException("Class " + newChild.getClass().getSimpleName() + " not accepted.");
		}

	}

	@Override
	public void neutralizeChild(PoshElement childNode) {
		if (this._actions.contains(childNode)) {
			if (this._actions.size() <= 1) {
				this.addTriggeredAction(new TriggeredAction("do_nothing"));
			}
			this._actions.remove(childNode);
			childNode.remove();
		}
	}

        public List<TriggeredAction> getTriggeredActions() {
            return Collections.unmodifiableList(_actions);
        }

    @Override
    public int compareTo(ActionPattern o) {
        return this.toString().compareTo(o.toString());
    }

    /**
     * Set timeout for AP.
     * @param t Timeout, not null
     */
    void setTimeout(SolTime t) {
        this.setNodeTimeoutAmmount(t.getCount());
        this.setNodeTimeoutUnits(t.getUnits().getId());
    }

    protected FormalParameters parameters = new FormalParameters();

    /**
     * Set the formal parameters of this 
     * @param parameters
     */
    protected void setParameters(FormalParameters parameters) {
        this.parameters = new FormalParameters(parameters);
    }

    /**
     * Get formal parametrs of this AP. Formal parameters contain map of parameterName-default value
     * this AP accepts.
     * @return
     */
    public FormalParameters getParameters() {
        return parameters;
    }
}
